require 'test/unit'
require 'lib/metrics_collector'

class TestMetricsCollector < Test::Unit::TestCase
  def setup
    @collector = MetricsCollector.new
  end
  
  def test_should_initialize_empty
    assert(@collector.metrics.empty?)
  end
  
  def test_should_accept_new_metric
    @collector.new_metric("Abc", 1.0)
    assert_equal(1, @collector.metrics.size)
    metric = @collector.metrics.first
    assert_equal("Abc", metric[:id])
    assert_equal(1.0, metric[:value])
  end
  
  def test_should_ignore_duplicate_metric
    @collector.new_metric("B", 2.0)
    @collector.new_metric("A", 1.0)
    @collector.new_metric("B", 2.0)
    assert_equal(2, @collector.metrics.size)
  end
  
  def test_should_create_if_adding_new_metric
    @collector.add_metric("B", 1.0)
    assert_equal(1, @collector.metrics.size)
    metric = @collector.metrics.first
    assert_equal("B", metric[:id])
    assert_equal(1.0, metric[:value])
  end
  
  def test_should_add_to_existing_id
    @collector.add_metric("B", 2.0)
    @collector.add_metric("B", 1.0)
    assert_equal(1, @collector.metrics.size)
    metric = @collector.metrics.first
    assert_equal("B", metric[:id])
    assert_equal(3.0, metric[:value])
  end

  def test_should_calculate_average
    @collector.new_metric("A", 1.0)
    @collector.new_metric("B", 2.0)
    @collector.new_metric("C", 3.0)
    assert_equal(2.0, @collector.average)
  end
  
  def test_should_calculate_std_dev
    @collector.new_metric("A", 1.0)
    @collector.new_metric("B", 2.0)
    @collector.new_metric("C", 3.0)
    assert_equal(1.0, @collector.std_dev)
  end
  
  def test_should_calculate_min
    @collector.new_metric("A", 0.23)
    @collector.new_metric("B", 1.0/3.0)
    @collector.new_metric("C", 3.0)
    assert_equal(0.23, @collector.min)
  end
  
  def test_should_calculate_max
    @collector.new_metric("A", 0.23)
    @collector.new_metric("B", 1.0/3.0)
    @collector.new_metric("C", 3.0)
    assert_equal(3, @collector.max)
  end
  
  def test_should_display_result
    @collector.new_metric("A", 0.5)
    @collector.new_metric("B", 2.0)
    @collector.new_metric("C", 3.0)
    @collector.add_metric("A", 0.5)
    
    assert_equal("# Average\tStdDev\tMin\tMax\tClasses: 3\n2.0\t1.0\t1.0\t3.0", @collector.result)
  end
  
  def test_should_ignore_test
    @collector.new_metric("Teste", 1.0)
    @collector.new_metric("teste", 2.0)
    @collector.new_metric("MeuTeste", 3.0)
    @collector.new_metric("MyTest", 4.0)
    @collector.new_metric("meu_testamento", 5.0)
    assert_equal(0, @collector.metrics.size)
  end
end
